#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmCollectionFacilityMaintain;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_Address : BaseTest 
	{
		private DataTable _dtAddress;
		private DataTable _dtFacility;
		private BOL.Address _tstAddress;

		[SetUp]
		protected void SetUp() 
		{			
			base.BaseSetUp();

			if (this.RefreshData)
			{
				this._dtAddress = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 *, RowVersion AS AddressRowVersion FROM " + TABLES.Address.TableName,  true);

				//*Facility Table
				this._dtFacility = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Facility WHERE AddressGuid IS NOT NULL",  true);

				this.RefreshData = false;
			}

			//Do it once for code reuse
			this._tstAddress = new BOL.Address();
			this._tstAddress.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.NewAddress);
		}

		[Test]
		public void Ctor_Blank_Pass()
		{
			Assert.AreEqual(string.Empty, this._tstAddress.AddressLine1, TABLES.Address.AddressLine1);
			Assert.AreEqual(string.Empty, this._tstAddress.AddressLine2, TABLES.Address.AddressLine2);
			Assert.AreEqual(string.Empty, this._tstAddress.AddressLine3, TABLES.Address.AddressLine3);
			Assert.AreEqual(string.Empty, this._tstAddress.City, TABLES.Address.AddressCity);
			Assert.AreEqual(string.Empty, this._tstAddress.StateCode, TABLES.Address.StateCode);
			Assert.AreEqual(string.Empty, this._tstAddress.ZipCode, TABLES.Address.AddressZipCode);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void Ctor_Blank_Fail()
		{
		}

		[Test]
		public void Ctor_DataRow_Pass()
		{
			BOL.Address testAddress = new BOL.Address(this._dtAddress.Rows[0]);

			Assert.AreEqual(this._dtAddress.Rows[0][TABLES.Address.AddressLine1].ToString(), testAddress.AddressLine1, TABLES.Address.AddressLine1);
			Assert.AreEqual(this._dtAddress.Rows[0][TABLES.Address.AddressLine2].ToString(), testAddress.AddressLine2, TABLES.Address.AddressLine2);
			Assert.AreEqual(this._dtAddress.Rows[0][TABLES.Address.AddressLine3].ToString(), testAddress.AddressLine3, TABLES.Address.AddressLine3);
			Assert.AreEqual(this._dtAddress.Rows[0][TABLES.Address.AddressCity].ToString(), testAddress.City, TABLES.Address.AddressCity);
			Assert.AreEqual(this._dtAddress.Rows[0][TABLES.Address.StateCode].ToString(), testAddress.StateCode, TABLES.Address.StateCode);
			Assert.AreEqual(this._dtAddress.Rows[0][TABLES.Address.AddressZipCode].ToString(), testAddress.ZipCode, TABLES.Address.AddressZipCode);
			Assert.AreEqual(false, testAddress.IsDirty, "IsDirty");
		}

		[Test]
		public void Ctor_DataRow_Fail()
		{
			BOL.Address tstAddress = new BOL.Address(this._dtAddress.NewRow());

			Assert.IsNotNull(tstAddress.AddressGuid, TABLES.Address.AddressGuid);
			Assert.AreEqual(string.Empty, tstAddress.City, TABLES.Address.AddressCity);
			Assert.AreEqual(string.Empty, tstAddress.StateCode, TABLES.Address.StateCode);
			Assert.AreEqual(string.Empty, tstAddress.ZipCode, TABLES.Address.AddressZipCode);

		}

		[Test]
		public void Ctor_Guid_Pass()
		{
			System.Guid facilityGuid = (System.Guid)this._dtFacility.Rows[0][TABLES.Facility.FacilityGuid];
			BOL.Address testAddress = new BOL.Address(facilityGuid);

			Assert.AreEqual((System.Guid)this._dtFacility.Rows[0][TABLES.Facility.AddressGuid], testAddress.AddressGuid, TABLES.Address.AddressGuid);
			Assert.AreEqual(false, testAddress.IsDirty, "IsDirty");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Guid_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			//AddressColumns aren't required, so filled in "nullable" columns with data for compare tests
			this._tstAddress.AddressLine1 = "1600 Pensylvania Ave";
			this._tstAddress.AddressLine2 = "Clinton's Desk";
			this._tstAddress.AddressLine3 = "ATTN: Monica";
			this._tstAddress.City = "Washington";
			this._tstAddress.StateCode = "DC";
			this._tstAddress.ZipCode = "12345";

			DataRow dtRow = this._tstAddress.LoadDataRowFromThis(this._dtAddress.NewRow());

			Assert.AreEqual(this._tstAddress.AddressGuid, (System.Guid)dtRow[TABLES.Address.AddressGuid], TABLES.Address.AddressGuid);
			Assert.AreEqual(this._tstAddress.AddressLine1, dtRow[TABLES.Address.AddressLine1].ToString(), TABLES.Address.AddressLine1);
			Assert.AreEqual(this._tstAddress.AddressLine2, dtRow[TABLES.Address.AddressLine2].ToString(), TABLES.Address.AddressLine2);
			Assert.AreEqual(this._tstAddress.AddressLine3, dtRow[TABLES.Address.AddressLine3].ToString(), TABLES.Address.AddressLine3);
			Assert.AreEqual(this._tstAddress.City, dtRow[TABLES.Address.AddressCity].ToString(), TABLES.Address.AddressCity);
			Assert.AreEqual(this._tstAddress.StateCode, dtRow[TABLES.Address.StateCode].ToString(), TABLES.Address.StateCode);
			Assert.AreEqual(this._tstAddress.ZipCode, dtRow[TABLES.Address.AddressZipCode].ToString(), TABLES.Address.AddressZipCode);

		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void LoadDataRowFromThis_Fail()
		{
		}

		[Test]
		public void GetEmptyAddressTable_Pass()
		{
			DataTable dtTest = BOL.Address.GetEmptyAddressTableSchema();

			Assert.IsNotNull(dtTest);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetEmptyAddressTable_Fail()
		{
		}

		[Test]
		public void AddressGuid_Pass()
		{
			System.Guid newGuid = System.Guid.NewGuid();

			this._tstAddress.AddressGuid = newGuid;

			Assert.AreEqual(newGuid, this._tstAddress.AddressGuid);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void AddressGuid_Fail()
		{
			this._tstAddress.AddressGuid = System.Guid.Empty;
		}

		[Test]
		public void IsNew_Pass()
		{
			bool tmpValue = !this._tstAddress.IsNew;

			this._tstAddress.IsNew = tmpValue;

			Assert.AreEqual(tmpValue, this._tstAddress.IsNew);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsNew_Fail()
		{
		}


		[Test]
		public void AddressLine1_Pass()
		{
			string tmpString = "".PadRight(10, 'X');	

			this._tstAddress.AddressLine1 = tmpString;
			Assert.AreEqual(tmpString, this._tstAddress.AddressLine1, "Address-HasColumn");

			//Test when column is missing
			DataTable dtTest = this._dtAddress.Copy();
			dtTest.Columns.Remove(TABLES.Address.AddressLine1);
			BOL.Address tstAddress = new BOL.Address(dtTest.Rows[0]);
			Assert.AreEqual(string.Empty, tstAddress.AddressLine1, "Address-NoColumn");

		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void AddressLine1_Fail1()
		{
			this._tstAddress.AddressLine1 = "".PadRight(256, 'X');	
		}

		[Test]
		public void AddressLine2_Pass()
		{
			string tmpString = "".PadRight(10, 'X');
			this._tstAddress.AddressLine2 = tmpString;

			Assert.AreEqual(tmpString, this._tstAddress.AddressLine2);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void AddressLine2_Fail()
		{
			this._tstAddress.AddressLine2 = "".PadRight(256, 'X');	
		}

		[Test]
		public void AddressLine3_Pass()
		{
			string tmpString = "".PadRight(10, 'X');
			this._tstAddress.AddressLine3 = tmpString;

			Assert.AreEqual(tmpString, this._tstAddress.AddressLine3);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void AddressLine3_Fail()
		{
			this._tstAddress.AddressLine3 = "".PadRight(256, 'X');	
		}

		[Test]
		public void City_Pass()
		{
			string tmpString = "".PadRight(10, 'X');

			this._tstAddress.City = tmpString;
			Assert.AreEqual(false, this._tstAddress.IsRuleBroken(BROKEN.City), "BrokenRule");

			//Test when column is missing
			DataTable dtTest = this._dtAddress.Copy();
			dtTest.Columns.Remove(TABLES.Address.AddressCity);
			BOL.Address tstAddress = new BOL.Address(dtTest.Rows[0]);
			Assert.AreEqual(string.Empty, tstAddress.City, "NoColumn");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void City_Fail()
		{
			this._tstAddress.City = "".PadRight(56, 'X');	
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void City_Fail2()
		{
			this._tstAddress.City = string.Empty;
			Assert.AreEqual(true, this._tstAddress.IsRuleBroken(BROKEN.City), "BrokenRule");

			//Crap on the set when column doesn't exist
			DataTable dtTest = this._dtAddress.Copy();
			dtTest.Columns.Remove(TABLES.Address.AddressCity);
			BOL.Address tstAddress = new BOL.Address(dtTest.Rows[0]);
			tstAddress.City = System.DateTime.Now.ToString();
		}

		[Test]
		public void StateCode_Pass()
		{
			this._tstAddress.StateCode= "".PadRight(2, 'X');
			Assert.AreEqual(false, this._tstAddress.IsRuleBroken(BROKEN.StateCode), "BrokenRule");

			//Test when column is missing
			DataTable dtTest = this._dtAddress.Copy();
			dtTest.Columns.Remove(TABLES.Address.StateCode);
			BOL.Address tstAddress = new BOL.Address(dtTest.Rows[0]);
			Assert.AreEqual(string.Empty, tstAddress.StateCode, "NoColumn");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void StateCode_Fail()
		{
			this._tstAddress.StateCode= "".PadRight(3, 'X');	
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void StateCode_Fail2()
		{
			this._tstAddress.StateCode= string.Empty;
			Assert.AreEqual(true, this._tstAddress.IsRuleBroken(BROKEN.StateCode), "BrokenRule");

			//Crap on the set when column doesn't exist
			DataTable dtTest = this._dtAddress.Copy();
			dtTest.Columns.Remove(TABLES.Address.StateCode);
			BOL.Address tstAddress = new BOL.Address(dtTest.Rows[0]);
			tstAddress.StateCode = "IL";
		}

		[Test]
		public void Zip_Pass()
		{
			this._tstAddress.ZipCode = "".PadRight(5, '6');
			Assert.AreEqual(false, this._tstAddress.IsRuleBroken(BROKEN.ZipCode), "BrokenRule");

			//Test when column is missing
			DataTable dtTest = this._dtAddress.Copy();
			dtTest.Columns.Remove(TABLES.Address.AddressZipCode);
			BOL.Address tstAddress = new BOL.Address(dtTest.Rows[0]);
			Assert.AreEqual(string.Empty, tstAddress.ZipCode, "NoColumn");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void ZipCode_Fail()
		{
			this._tstAddress.ZipCode = "".PadRight(11, '6');
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void ZipCode_Fail2()
		{
			//Crap on the set when column doesn't exist
			DataTable dtTest = this._dtAddress.Copy();
			dtTest.Columns.Remove(TABLES.Address.AddressZipCode);
			BOL.Address tstAddress = new BOL.Address(dtTest.Rows[0]);

			tstAddress.ZipCode = "60185";
		}
	}
}
#endif



